<?php
/******
 *Class Name:WowMultiXml
 *Author: Tracy Ridge
 *Version: 1.0
 *Updated 5th February 2018
 *Author URL:  https:www.worldoweb.co.uk/
 *Page URL: https://wp.me/poe8j-32l
 *Description:  Display multiple RSS feeds on a website.
*******/

class WowMultiXml{

	//Default Values - Overidden by constructor
	private $options = array(
					 'url'=>'http://feeds.skynews.com/feeds/rss/world.xml',
					 'maxItems' =>10,
					 'date_format' => 'jS F, Y',
					 'sort'=> array('sort_order' => 'new_first')
			  );

		/**
		 * Constructor
		 * @param array|string $urls		Urls of feeds
		 * @param int|string $maxitems 	Maximum items to display
		 * @param string $date_format 	Date Format
		 * @param array $sortfeed    		Sort feed asc|des
		 */
		public function __construct($urls, $maxitems, $date_format='',$sortfeed)
			{
				//Checks to see you have entered a string or an array - Defaults to options array
				switch ($urls) {
					case is_array($urls):
						$this->urls = $urls;
						break;
					case is_string($urls):
						$this->urls=array($urls);
						break;
					default:
						$this->urls = array($this->options['url']);
						break;
				}
				if (is_string($date_format)){$this->date_format = $date_format;}
				else{$this->date_format = $this->options['date_format'];}

				switch ($sortfeed) {
					case is_array($sortfeed):
						$this->sortfeed = $sortfeed;
						break;
					default:
						$this->sortfeed = $this->options['sort'];
						break;
				}
				//Checks for intval (string or int)
				if (intval($maxitems)){$this->maxitems = $maxitems;}
				else{$this->maxitems = $this->options['maxItems'];}
			}

		/**
		 *   __tostring
		 *   Converts obj to a string
		 *
		 * @return string displays rss feed
		 */

			public function __toString(){
				return $this->displayRss();
			}

			/**
			 * Converts published date to a unix timestamp via callback
			 * @author Tracy Ridge
			 * @param  array $item
			 * @return array
			 */
			private function convertTimestamp(&$item){

				foreach ($item as $key=> $value) {
						$value->pubDate = strtotime(trim($value->pubDate));

					}
					return $item;
				}

			/**
			 * sortDate converts the timestamp via convertTimestamp then sorts the feed using specified sort order
			 * @author Tracy Ridge
			 * @param  array $xml
			 * @return array
			 */
			public function sortDate($xml){
				//converts the timestamp for sorting
				$array = $this->convertTimestamp($xml);

				$sortfeed = $this->sortfeed;

				//sort the feed using compare function
					switch($sortfeed['sort_order'])
					{
						case 'old_first':
							usort($array, function ($a, $b) { return ($a->pubDate - $b->pubDate); });

							break;

						case 'new_first':
							usort($array, function ($a, $b) { return ($b->pubDate - $a->pubDate); });

							break;
					}

				return $array;
			}

			/**
			 * Fetches feeds using simplexml_load_file, uses sortDate to organise
			 * @author Tracy Ridge
			 * @return array
			 */

			public function getFeed(){
				//Get Urls from array and load data into an array
				foreach($this->urls as $url){
					$array[] = simplexml_load_file($url, "SimpleXMLElement", LIBXML_NOCDATA);
				}
				//Create new array
				$a = array();
				//loop through array
				foreach ($array as &$x) {
					//loop through each channel
					foreach ($x->channel->item as $item){
							//add each channel to an array
								$a[]=$item;
					}
				}
				$a = $this->sortDate($a);

				return $a;
			}

		/**
		 * Display RSS fetches the feed via getFeed and outputs it to the browser via toString
		 * @author Tracy Ridge
		 * @return string
		 */

		public function displayRss(){
			$feed = '';

			$xml = $this->getFeed();

			$feed.= '<div class="wow_feed">';

				$i=-1;

				foreach ($xml as $value) {
					if (++$i == $this->maxitems) break;//breaks when max-items is reached

					$url 		= $value->link;
					$title 	= $value->title;
					$desc 	= $value->description;

					if($this->date_format != ''){
						$date = $value->pubDate;
						$d = $this->date_format;
						$date = date($d,intval($date));//Convert to proper date
					}

					if (isset($value->category) && !empty($value->category)):
						$cat = $value->category;
					else:
						$cat = null;
					endif;

						$feed .= "<h2><a href='{$url}' title='{$title}'>{$title}</a></h2>";

					if (isset($cat) && !empty($cat)){
						$feed .= "<div class='cat'>{$cat}</div>";
					}
					if (!empty($date)){
						$feed .= "<div class='date'>{$date}</div>";
					}
						$feed .= "<div class='desc'>{$desc}</div>";

			}
			return $feed;
		}
}
